-- ==============================================================
-- ------- Trashcan 2.20R -------
-- File: tcoptions.lua
-- Modified date:    2011-03-04T04:33:27Z
-- Author:	MSaint (current version: msaint)
-- Desc:    GUI Configuration Options for Trashcan
-- ==============================================================

local _, _, OUR_VERSION = string.find("2.20R", "([%d\.]+)")
OUR_VERSION = tonumber(OUR_VERSION) or 2
local DEBUG = nil
local Debug = DEBUG and function(s) DEFAULT_CHAT_FRAME:AddMessage(s, 1, 0, 0) end or function() return end  

-- Create the addon object if needed  (note: only Trashcan.lua should actually set the version!)
if (Trashcan and Trashcan.Version and Trashcan.Version > OUR_VERSION) then return end
Trashcan = Trashcan or {}
local tc, tcDB = Trashcan, nil --tcDB will be pointed at TrashcanDB just before options are registered

-- Set up for future localization
local L = tc.L or {}
setmetatable( L, { __index = function(t, text) return text end })

-- Make library calls local
local type, tostring, next, pairs, ipairs = type, tostring, next, pairs, ipairs  
local string, table, tinsert = string, table, tinsert

-- Make api calls local
local GetItemInfo = GetItemInfo

-- Libraries
local cfg = LibStub("AceConfig-3.0")
local gui = LibStub("AceConfigDialog-3.0")
tc.GuiUpdate = LibStub("AceConfigRegistry-3.0").NotifyChange

-- Static Dialogs 
StaticPopupDialogs["CONFIRM_REMOVE_TRASHLIST_ITEM"] = {
	text = L["Are you sure you want to remove %s from the Trashcan list?"],
	button1 = YES,
	button2 = NO,
	OnAccept = function (self, data)
         if (data and type(data) == "string") then
            tcDB.TrashList[data] = nil
            tc:GuiMakeTrashlist()
            if (not tcDB.Silent) then
               DEFAULT_CHAT_FRAME:AddMessage(L["TC: "] .. data .. L[" removed from Trashcan list!"], true)
            end
         end
      end,
	OnCancel = function (self) end,
	hideOnEscape = 1,
	timeout = 0,
	exclusive = 1,
	whileDead = 1,
}

StaticPopupDialogs["CANNOT_REMOVE_TRASHLIST_ITEM"] = {
	text = "%s",
	button1 = YES,
	OnAccept = function (self) end,
	hideOnEscape = 1,
	timeout = 0,
	exclusive = 1,
	whileDead = 1,
}


tc.options = {
   type = "group",
   childGroups = "tab",
   args = {
      mainCfgTab = {
         name = L["Configuration"],
         type = "group",
         order = 1,
         args = {
            enable = {
               order = 1,
               width = "full",
               name = L["Destroy Junk"],
               desc = L["Destroy all grays and items in the trash list when the conditions below are met."],
               type = "toggle",
               set = function(info, val) tc:Enable(val) end,
               get = function(info) return tcDB.Enabled end,
            },
            sell = {
               order = 3,
               width = "full",
               name = L["Sell Junk"],
               desc = L["Sell all undestroyed grays and items in the trash list whenever you interact with a vendor.\n(If 'Destroy Junk' is set, this option will still sell junk that did not meet the below conditions and was not destroyed)"],
               type = "toggle",
               disabled = function() return tcDB.Enabled and not tcDB.useMaxVal and not tcDB.useMinFree end,
               set = function(info, val) tcDB.Sell = val end,
               get = function(info) return tcDB.Sell end,
            },
            silent = {
               order = 5,
               width = "full",
               name = L["Silent Mode"],
               desc = L["Suppresses most of Trashcan's chat messages."],
               type = "toggle",
               set = function(info, val) tcDB.Silent = val end,
               get = function(info) return tcDB.Silent end,
            },
            separator1 = {
               order = 7,
               disabled = true,
               width = "full",
               name = "\n",
               type = "description",
            },
            useMaxval = {
               order = 11,
               disabled = function() return not tcDB.Enabled end,
               width = "full",
               name = L["Don't destroy junk unless it's worth less than a certain value"],
               desc = L["Destroy only items worth less than a certain amount (per stack)."],
               type = "toggle",
               set = function(info, val) tcDB.useMaxVal = val end,
               get = function(info) return tcDB.useMaxVal end,
            },
            maxval = {
               order = 21,
               hidden = function() return not (tcDB.Enabled and tcDB.useMaxVal) end,
               width = "full",
               name = function () return L["Sell junk worth less than "] .. tc:GSCString(tcDB.MaxVal) end,
               desc = function () return L["Only items that vendor for less than "].. tc:GSCString(tcDB.MaxVal) .. L[" will be destroyed."] end,
               type = "range",
               min = 10,
               max = 1000000,
               softMin = 1,
               softMax = 20000,
               step = 10,
               bigStep = 10,
               set = function(info, val) tcDB.MaxVal = val end,
               get = function(info) return tcDB.MaxVal end,      
            },
            separator2 = {
               order = 26,
               disabled = true,
               width = "full",
               name = "\n",
               type = "description",
            },
            useMinfree = {
               order = 31,
               disabled = function() return not tcDB.Enabled end,
               width = "full",
               name = L["Destroy junk only when I am low on space"],
               desc = L["Do not destroy junk right away, wait til I am low on space."],
               type = "toggle",
               set = function(info, val) tcDB.useMinFree = val end,
               get = function(info) return tcDB.useMinFree end,
            },
            minfree = {
               order = 41,
               hidden = function() return not (tcDB.Enabled and tcDB.useMinFree) end,
               width = "full",
               name = L["Slots to try to keep free"],
               desc = L["Number of bag slots to try to keep free by destroying junk."],
               type = "range",
               min = 1,
               max = 300,
               softMin = 1,
               softMax = 40,
               step = 1,
               bigStep = 1,
               set = function(info, val) tcDB.MinFree = val end,
               get = function(info) return tcDB.MinFree end,      
            },
         },
      },
      infoTab = {
         name = L["History"],
         type = "group",
         order = 3,
         args = {
            separator1 = {
               order = 11,
               disabled = true,
               width = "full",
               name = "\n",
               type = "description",
            },
            separator2 = {
               order = 21,
               disabled = true,
               width = "full",
               name = "\n",
               type = "description",
            },
            separator3 = {
               order = 53,
               disabled = true,
               width = "full",
               name = "\n",
               type = "description",
            },
            countToDate = {
               order = 60,
               disabled = true,
               width = "full",
               fontSize = "medium",
               name = function () return L["Total number of items destroyed:      "] .. tostring(tcDB.Count) end,
               type = "description",
            },
            valueToDate = {
               order = 61,
               disabled = true,
               width = "full",
               fontSize = "medium",
               name = function () return L["Total value of all items destroyed:     "] .. tc:GSCString(tcDB.Value) end,
               type = "description",
            },
            separator4 = {
               order = 66,
               disabled = true,
               width = "full",
               name = "\n",
               type = "description",
            },
            separator5 = {
               order = 73,
               disabled = true,
               width = "full",
               name = "\n",
               type = "description",
            },
            SoldCountToDate = {
               order = 81,
               disabled = true,
               width = "full",
               fontSize = "medium",
               name = function () return L["Total number of items sold:      "] .. tostring(tcDB.SoldCount) end,
               type = "description",
            },
            SoldValueToDate = {
               order = 81,
               disabled = true,
               width = "full",
               fontSize = "medium",
               name = function () return L["Total value of all items sold:     "] .. tc:GSCString(tcDB.SoldValue) end,
               type = "description",
            },
            separator6 = {
               order = 86,
               disabled = true,
               width = "full",
               name = "\n",
               type = "description",
            },
            separator7 = {
               order = 87,
               disabled = true,
               width = "full",
               name = "\n",
               type = "description",
            },
            separator8 = {
               order = 88,
               disabled = true,
               width = "full",
               name = "\n",
               type = "description",
            },
            clearHistory = {
               order = 91,
               name = L["Clear History"],
               desc = L["Forget the total count and value of all items destroyed and sold to date (reset it to zero)."],
               type = "execute",
               func = function() tcDB.Count, tcDB.SoldCount, tcDB.Value, tcDB.SoldValue = 0, 0, 0, 0 end,
            },
         },
      },   
   },
}

local currentItem, currentSelected, guiTrashlist
local trashlistTab = {
   name = L["Trash List"], 
   type = "group",
   order = 2,
   args = {
      dropTarget = {
         order = 10,
         width = "full",
         type = "input",
         dialogControl = "MSDropTarget",
         name = function ()
               local itemName = currentItem and GetItemInfo(currentItem) or L["Drag item onto button to add to trash list."]
               return itemName
            end,
         desc = "#ItemToolTip#",
         validate = function (info, value)
               local _, _, type = string.find(value, "(%a+)")
               return type == "item" and true or L["Not an item."]
            end,
         get = function () return currentItem end,
         set = function (info, value) currentItem = value end,
      },
      addButton = {
         name = L["Add to Trash List"],
         order = 20,
         width = "full",
         type = "execute",
         func = function ()
               if currentItem and tc:AddToDeleteList(currentItem) then
                  currentItem = nil
               end
            end,
      },
      trashlist = {
         name = L["Click below to remove an item from the trash list:"],
         order = 30,
         type = "group",
         inline = true,
         args = {
            theList = {
               type = "select",
               dialogControl = "MSItemSelectList",
               name = "Anyname", -- This is not actually used by the widget atm
               desc = "#ItemTooltip#",
               width = "full",
               values = function () return guiTrashlist or tc:GuiMakeTrashlist() end,
               get = function () return currentSelected end,            
               set = function (info, key) tc:GuiRemoveFromTrashlist(key)
                  Debug("Selected has been set to: " .. key)
                  currentSelected = key end,            
            },
         },
      },  
   },                                                            
}
tc.options.args.trashlistTab = trashlistTab


function tc:GuiMakeTrashlist()
   local dbTrashlist = tcDB.TrashList
   if ((not dbTrashlist) or (not next(dbTrashlist))) then
      Debug("The DB Trashlist is empty.  Returning {}.")
      guiTrashlist = {}
      tc:GuiUpdate(tc.OptionsName)
      return guiTrashlist
   end
   local list = {}   
   local shortItemString
   for name, _ in pairs(dbTrashlist) do
      --Debug("next name in dbTrashlist is: " .. name)
      tinsert(list, name)
   end
   -- give list a sensible order so user doesn't go crazy in a long list
   table.sort(list)
   guiTrashlist = {}
   for i, name in ipairs(list) do
      shortItemString = "item:" .. tostring(dbTrashlist[name])
      guiTrashlist[i] = shortItemString
      --Debug("guiTrashlist["..tostring(i).."] = "..shortItemString)
   end
   tc:GuiUpdate(tc.OptionsName)
   return guiTrashlist
end


function tc:GuiRemoveFromTrashlist(key)
   local itemstring = guiTrashlist and guiTrashlist[key] 
   if itemstring then
      local trashlist = tcDB.TrashList
      local name = GetItemInfo(itemstring)
      if (name and trashlist and trashlist[name]) then
         local popup = StaticPopup_Show("CONFIRM_REMOVE_TRASHLIST_ITEM", name)
         if popup then
            popup.data = name
         end         
      elseif name then
         StaticPopup_Show("CANNOT_REMOVE_TRASHLIST_ITEM", name .. L[" was not found in the Trashcan list."])
      else
         StaticPopup_Show("CANNOT_REMOVE_TRASHLIST_ITEM", name .. L[" is not a valid item."])
      end
   end
end

function tc:LoadOptions()
   if (cfg and gui) then 
      -- Take the opportunity to make a local reference to the DB
      tcDB = TrashcanDB
      -- Add options to Bliz addon config frame
      cfg:RegisterOptionsTable(tc.OptionsName, tc.options)
      gui:AddToBlizOptions(tc.OptionsName)
   end
end
