-- ==============================================================
-- ------- Trashcan 2.20R -------
-- File: trashcan.lua
-- Modified date:    2012-03-29T18:47:41Z
-- Author:	Wortahk (current version: msaint)
-- Desc:    Deletes your junk.
-- New in this version:
--          Added GUI options to Bliz Addon options
--          Added maxval threshold, will only sell items worth less
-- New in previous version:
--          Added various slash commands (see below)
--          Added delete list for non-grey items
--          Added third button to Bliz item deletion popup to
--             add item to TC delete list
--          Added saved settings (enabled/disabled and delete list)
--          Added bagspace threshold, to start deleting only when
--             space is low.
-- Usage:  	/trash | /tc [destroy [on | off] | sell [ on | off] | config | status | list 
--            | remove [<itemname> | ALL] | minfree [ off | <numslots> ]
--            | maxval [ off | <copper> ] | config | purge | value | test]
--             - with no argument, displays options
--             - on or off enables or disables trashcan
--             - config opens gui options frame
--             - status displays the enabled / disabled status
--             - list prints the items in the TC delete list
--             - remove takes an item out of the TC delete list
--             - minfree tells TC not to delete items until only <numslots>
--                are free
--             - maxval tells TC to only delete items worth less than <copper>
--                (per stack, if stackable).
--             - config opens the GUI configuration options window
--             - purge destroys grey / saved items even if the minfree has not
--                 been reached.
--             - value prints the value of items deleted by Trashcan to date
--             - test  prints a list of the items that would be
--                destroyed if trashcan were enabled
-- ==============================================================

local OUR_NAME = "Trashcan"
local _, _, OUR_VERSION = string.find("2.20R", "([%d\.]+)")
OUR_VERSION = tonumber(OUR_VERSION) or 2
local DEBUG = nil
local Debug = DEBUG and function(s) DEFAULT_CHAT_FRAME:AddMessage(s, 1, 0, 0) end or function() return end  

-- Magic Numbers
local MIN_DESTROY_INTERVAL = 1

-- Our Addon Object
if (Trashcan and Trashcan.Version and Trashcan.Version > OUR_VERSION) then return end
Trashcan = Trashcan or {}
local tc = Trashcan
tc.AddonName = OUR_NAME
tc.Version = OUR_VERSION
tc.OptionsName = OUR_NAME
tc.TrashlistOptionsName = OUR_NAME .. "-Trashlist"
tc.currentItem = nil
tc.Events = tc.Events or {}

-- Set up for future localization
local L = tc.L or {}
setmetatable( L, { __index = function(t, text) return text end })

-- Make library calls local
local type, tostring, tonumber, next, pairs, ipairs, setmetatable = type, tostring, tonumber, next, pairs, ipairs, setmetatable 
local math, string, table, tinsert = math, string, table, tinsert 

-- Make api calls local
local GetItemInfo = GetItemInfo
local GetContainerNumSlots = GetContainerNumSlots 
local GetContainerNumFreeSlots = GetContainerNumFreeSlots
local GetContainerItemInfo = GetContainerItemInfo
local GetContainerItemID = GetContainerItemID
local PickupContainerItem = PickupContainerItem 
local CursorHasItem = CursorHasItem
local GetCursorInfo = GetCursorInfo
local DeleteCursorItem = DeleteCursorItem

-- Libraries

-- Default Settings
local defaultDB = {
   Version = OUR_VERSION,
   Enabled = false,
   TrashList = {},
   useMinFree = true,
   MinFree = 5,
   useMaxVal = true,
   MaxVal = 10000,
   Value = 0,
   Count = 0,
   Silent = false,
   Sell = false,
   SoldValue = 0,
   SoldCount = 0,
}
local tcDB = defaultDB


-- Locals
local events = tc.Events

-- Local functions
local function checkVersion()
   local db = TrashcanDB
   if (db and (not db.Version or db.Version < OUR_VERSION)) then
      db.Version = OUR_VERSION
      -- This is an old DB, so update it.
      -- (so far, all we changed is handling of MinFree and MaxVal)
      if (db.useMinFree == nil) then
         db.useMinFree = db.MinFree and true or false
      end
      if (db.useMaxVal == nil) then
         db.useMaxVal = db.MaxVal and true or false
      end
   end
end

local function chatMsg(s, force, r, g, b)
   if (force and type(force) ~= "boolean") then
      r, g, b, force = force, r, g, nil
   end 
   if (not tcDB.Silent or force) then
      DEFAULT_CHAT_FRAME:AddMessage(s, r, g, b)
   end
end


local function getFreeBagSlots()
   local freeslots = 0
   for i=0, 4 do
      freeslots = freeslots + GetContainerNumFreeSlots(i)
   end
   return freeslots
end

local function pricecomp(a, b)
   if (a and a.value and b and b.value) then
      return (a.value < b.value)
   end
end

local function destroyJunk(listonly)
   listonly = (listonly == true) or false -- corrects problem when called by timed event
   if (not listonly and not tcDB.Enabled) then return end
   local maxToDel = (tcDB.useMinFree and not listonly) and tcDB.MinFree - getFreeBagSlots()
   if (maxToDel and maxToDel < 1) then return end
   local prefix = listonly and L["TC: Destroyable: "] or L["TC: Destroyed: "]
   local count, icount, link, name, quality, stacksize, value, stackval, itemId = 0
   local delList = {}
   local item = {}
   for bag = 0,4 do
      for slot = 1,GetContainerNumSlots(bag) do
         itemId = GetContainerItemID(bag, slot)
         _, icount, _, _, _, _, link = GetContainerItemInfo(bag, slot)
    		if (link) then
    		   name, _, quality, _, _, _, _, stacksize, _, _, value = GetItemInfo(itemId)
    			stackval = value * stacksize
            if (quality == 0 and (not tcDB.useMaxVal or stackval < tcDB.MaxVal) or tcDB.TrashList[name]) then
               item = {id = itemId, value = value, bag = bag, slot = slot, link = link, count = icount}
               tinsert(delList, item)
    		   end
    	   end
   	end
   end
   if (#delList > 0) then
      local oType, recheckId
      local numToDel = (maxToDel and maxToDel < #delList) and maxToDel or #delList
      table.sort(delList, pricecomp)
      for i = 1, numToDel do
         item = delList[i]
         if (item and item.link) then
            if (not listonly) then
               PickupContainerItem(item.bag, item.slot)
					-- Check that the item on the cursor is the item we wanted to destroy
					oType, recheckId = GetCursorInfo()
               if (oType == "item" and recheckId == item.id) then
                  DeleteCursorItem()
                  tcDB.Value = tcDB.Value + item.value * item.count
                  tcDB.Count = tcDB.Count + item.count
                  chatMsg(prefix .. item.link .. "   X " .. tostring(item.count), 1, 0, 0)
               end
            else -- for list only we don't confirm the item on the cursor
               chatMsg(prefix .. item.link .. "   X " .. tostring(item.count), 1, 0, 0)
               count = count + item.count
            end
         end
      end
   end
   if (listonly) then
      local message
      if (count == 0) then
         message = L["Nothing to destroy."]
      else
         message = L["There are "] .. count .. L[" items to destroy."]
	   end
      chatMsg(L["TC: "] .. message)
	else
	   tc:GuiUpdate(tc.OptionsName)
	end
end

local scheduleDestroy
do
   local destroyHandle
   scheduleDestroy = function ()
      -- Make sure we wait our MIN_DESTROY_INTERVAL before trying to destroy.
      -- LOOT_CLOSED and item being in inventory are not simultaneous.
      events:CancelTimedCallback(destroyHandle)
      destroyHandle = events:SetTimedCallback(destroyJunk, MIN_DESTROY_INTERVAL)
   end
end


local function parseCopper(cpval)
   local g = math.floor(cpval/10000)
   local s = math.floor((cpval - g*10000)/100)
   local c = math.floor(cpval - g*10000 - s*100)
   local gscString = tostring(c) .. "c"
   if (s > 0 or g > 0) then
      gscString = tostring(s) .. "s " .. gscString
   end
   if (g > 0) then
      gscString = tostring(g) .. "g " .. gscString
   end
   -- returning the string, but also the values, in case they were ever needed, which I doubt.
   return gscString, g, s, c
end

local function printStatus()
   local enabT = tcDB.Enabled and L["ENABLED"] or L["DISABLED"]
   local sellEnabT = tcDB.Sell and L["ENABLED"] or L["DISABLED"]
   local threshT = tcDB.useMinFree and tostring(tcDB.MinFree) or L["DISABLED"]
   local maxvalT = tcDB.useMaxVal and parseCopper(tcDB.MaxVal) or L["DISABLED"]
   local totalvalT = parseCopper(tcDB.Value)
   chatMsg(L["TC: Destroying junk is "] .. enabT, true)
   chatMsg(L["TC: Selling is "] .. sellEnabT, true)
   chatMsg(L["TC: Free bag slot threshold is "] .. threshT, true)
   chatMsg(L["TC: Max value threshold (per stack) is "] .. maxvalT, true)
   chatMsg(L["TC: The vendor value of all items ever deleted by Trashcan is "] .. totalvalT, true)
end

local function freshenItemData(List)
   -- If an item is looked up for the first time since entering world,
   -- GetItemData() will return nil for info that has not yet arrived
   -- locally.  If we do this at the start, it should all be present
   -- whenever it is needed. New items won't be a problem because the
   -- data is already there when the item is added.
   for _, id in pairs(tcDB.TrashList) do
      local _, link = GetItemInfo(id)
   end
end

local function fixPopup(doIt)
   local delPopup = StaticPopupDialogs["DELETE_ITEM"]
   if (doIt) then
      delPopup.text = DELETE_ITEM .. "\n\n"
	   delPopup.button3 = "Trashcan"
      delPopup.OnAlt = function (self)
         tc:AddCursorToDeleteList()
      end
   else
      delPopup.text = DELETE_ITEM
	   delPopup.button3 = nil
      delPopup.OnAlt = nil
   end
end

local function enable(doIt)
   tcDB.Enabled = doIt
   if (doIt) then
      events:RegisterEvent("LOOT_CLOSED")
      tc:GuiUpdate(tc.OptionsName)
   else
      events:UnregisterEvent("LOOT_CLOSED")
      tc:GuiUpdate(tc.OptionsName)
   end
end

-- Events / Handlers

function events:LOOT_CLOSED(...)
	-- Use OnUpdate to schedule a destroyJunk
   --Debug("Received LOOT_CLOSED event")
   scheduleDestroy()
	return
end

function events:ADDON_LOADED(...)
   local addonName = ...
   if (addonName == OUR_NAME) then
      -- This only needs to happen once
      events:UnregisterEvent("ADDON_LOADED")
      -- Use default settings if saved variables (TrashcanDB) are not there
      if (TrashcanDB) then
         checkVersion()
         tcDB = TrashcanDB
	   else
	      TrashcanDB = tcDB
      end
	   -- Set the defaults as a metatable to the settings, in case any are not there (or new)
      setmetatable(tcDB, { __index = function(l, index) return defaultDB[index] end})
      -- Make sure the client loads itemdata for everything in our TrashList
      freshenItemData(tcDB.TrashList)
      -- Add Slash Commands
      -- First check if /tc is taken, as its only two letters (ok, 26^2/numAddons chance, but still ...)
      local usetc, n, cmd = true
      for addon, _ in pairs(SlashCmdList) do
         n = 1
         repeat
            cmd = _G["SLASH_"..addon..tostring(n)]
            if (cmd and cmd == "/tc") then
               usetc = nil
               break
            end
            n = n + 1
         until (not cmd)
         if (not usetc) then break end
      end
      SlashCmdList["TRASHCAN"] = function (...) tc:cmdParser(...) end
      SLASH_TRASHCAN1 = "/trash"
      if usetc then
         SLASH_TRASHCAN2 = "/tc"
         chatMsg(L["TC: Trashcan loaded! Type '/tc' to list options."])
      else
         chatMsg(L["TC: Trashcan loaded! '/tc' is in use by another addon, use '/trash' instead."], true)
      end
      -- Add options to Bliz addon config frame
      tc:LoadOptions()
      -- Start up the merchant seller module
      tc.Merchant:OnAddonLoaded()
      -- Set popup, and Enable (or not ...)
      fixPopup(true)
      enable(tcDB.Enabled)
      -- We're done with this method forever
      events.ADDON_LOADED = nil
   end
end

do
   --** -----Set up event handling, timers, and triggered callbacks. ----- **
   --**
   --**This code is re-usable.  A table 'events' should be defined at file
   --**scope, i.e. 'local events = {}'.  This section will add the following
   --**methods to the table:
   --**  events:RegisterEvent(eventname)
   --**  events:UnregisterEvent(eventname) -- returns handle used to cancel
   --**  events:SetTriggeredCallback(name, checkFunc, actionFunc, data, repeating)
   --**  events:CancelTriggeredCallback(handle)
   --**  events:SetTimedCallback(func, delay, data, repeating) -- returns handle used to cancel
   --**  events:CancelTimedCallback(handle)

   --**
   --**The following lines should be changed to be specific to the Add-On
   local initialLoadEvent = "ADDON_LOADED"
   local eventFrameName = "TrashcanMainFrame"
   --**

   --**There MUST be a table 'events', and it should not be global!
   if not (events and type(events) == "table" and events ~= _G[events]) then
      error("Table 'events' must be defined at the file level.")
   end

   --Create the frame if it doesn't exist, as well as the locals we need
   local eFrame = _G[eventFrameName] or CreateFrame("Frame", eventFrameName, UIParent)
   local callbacks = {}
   local timeSinceStart = 0
   
   --No need for OnUpdate until a timer / trigger is set
   eFrame:Hide()
   
   --Registered events will be redirected to events:EVENT_NAME(...)
   eFrame:SetScript("OnEvent", function(self, event, ...)
         events[event](events, ...)
      end)
   
   eFrame:RegisterEvent(initialLoadEvent)

   local function OnUpdate(self, elapsed)
      timeSinceStart = timeSinceStart + elapsed
      for handle, tEvent in pairs(callbacks) do 
         local data = tEvent.data 
         if (tEvent.check and tEvent.check(timeSinceStart - tEvent.eventStart, data)) then
            Debug("We are now trying to execute the action for " .. tEvent.name)
            tEvent.action(data)
            if (not tEvent.repeating) then
               callbacks[handle] = nil
               if not next(callbacks) then
                  eFrame:Hide()
               end
            end
         end
      end
   end
   eFrame:SetScript("OnUpdate", OnUpdate)
   
   function events:SetTriggeredCallback(eName, check, action, data, repeating)
   --This function is more generalized than needed in this addon at this time,
   --but I have it written, the cost is not high, and it could be useful later.
   --Params:
   --    eName --A name for the event/callback.  If unique, can be used to cancel the event.
   --    check --function(elapsed, data) used to check if a callback should trigger.
   --    action --function(data) will be called back if check returns true.
   --    data --arbitrary data to be passed to both check and action.
   --    repeating --If true, the event/callback will not be removed when triggered
   --          and until canceled will keep firing whenever check returns true.
   --Returns:
   --    handle --A unique identifier for this event/callback.  This should be used
   --          when canceling an event, as this is more efficient than using the
   --          event name.    
      Debug("Entered SetTriggeredCallback")
      if (type(check) ~= "function" or type(action) ~= "function") then
         return
      else
         Debug("We are setting up event : " .. eName)
         --if repeating then Debug("Event " .. eName .. " is repeating") end
         local tEvent = {
            eventStart = timeSinceStart,
            name = eName,
            check = check,
            action = action,
            data = data,
            repeating = repeating,
         }
         local handle = {}
         callbacks[handle] = tEvent
         eFrame:Show()
         return handle
      end                     
   end

   function events:SetTimedCallback(func, delay, data, repeating)
   --A more convenient way to set a simple timed callback. Can only be canceled
   --using the handle.
      local check = function(elapsed)
      	if (elapsed > delay) then
      		return true
      	end
      end
      return events:SetTriggeredCallback("generic_timer", check, func, data, repeating)  
   end
   
   function events:CancelTriggeredCallback(handle)
   --Parameter can be the handle returned when creating the triggered event,
   --or else the name given as the first parameter when creating the event.
   --It is less efficient and more dangerous to use the name, as the table of
   --callbacks must be searched, and all matching events will be canceled.
      if (type(handle) == "string") and callbacks then
         -- What, you can't be bothered to store the index?
         local eName = handle
         for h, tEvent in pairs(callbacks) do
            if (tEvent.name == eName) then
               callbacks[h] = nil
               if not next(callbacks) then
                  eFrame:Hide()
               end
            end
         end
      end
      if (callbacks and handle and callbacks[handle]) then
         --Debug("Canceling timed event : " .. callbacks[index].name)
         callbacks[handle] = nil
         if not next(callbacks) then
            eFrame:Hide()
         end
      end
   end
   events.CancelTimedCallback = events.CancelTriggeredCallback
   
   --Since the handlers are on the 'events' object, lets make registering intuitive
   function events:RegisterEvent(...)
      eFrame:RegisterEvent(...)
   end
   
   function events:UnregisterEvent(...)
      eFrame:UnregisterEvent(...)
   end 
end


-- Class methods
function tc:Enable(doIt, loud)
   enable(doIt)
   if (loud) then
      local tcStatus = tcDB.Enabled and L["ENABLED"] or L["DISABLED"]
      chatMsg(L["TC: Trashcan is "] .. tcStatus .. ".", true)
   end
end


function tc:cmdParser(optionTxt)
   -- Really, most of this is sort of unecessary with the ui page added, will probably remove some options from help,
   -- although there is no real reason to actually remove them from the code
   if (not type(optionTxt) == "string" or optionTxt == "" or optionTxt == L["help"]) then
      chatMsg(L["TC: Options: destroy | sell | config | minfree | maxval | status | list | remove | purge | value | test"], true)
      return
   end
   local listonly = false
   local option, arg2 = optionTxt:match("^(%S*)[%s,]*(.-)$")
   if (option == L["status"]) then
		printStatus()
		return
	elseif (option == L["config"]) then
      InterfaceOptionsFrame_OpenToCategory(tc.OptionsName) 
      return
   elseif (option == L["value"]) then
      local totalvalT = parseCopper(tcDB.Value)
      chatMsg(L["TC: The vendor value of all items ever deleted by Trashcan is "] .. totalvalT, true)
      return
   elseif (option == L["list"]) then
      tcDB.Silent = false
      if (next(tcDB.TrashList)) then
         chatMsg(L["TC: Trashcan will always destroy these items:"], true)
         for _, id in pairs(tcDB.TrashList) do
            local _, link = GetItemInfo(id)
            chatMsg(link)
         end
      else
         chatMsg(L["TC: Your Trashcan list is empty."], true)
      end
      return
   elseif (option == L["maxval"]) then
      local maxVal, maxvalT = tonumber(arg2)
      if (arg2 == L["off"]) then
         tcDB.useMaxVal = false
         chatMsg(L["TC: Trashcan will delete items regardless of value."], true)
      elseif (maxVal and maxVal > 0) then
         tcDB.useMaxVal = true
         tcDB.MaxVal = maxVal
         maxvalT = parseCopper(maxVal)
         chatMsg(L["TC: Trashcan will only destroy items worth less than "] .. maxvalT .. L[" per stack."], true)
      else
         chatMsg(L["TC: Valid options are /tc maxval [ <value> | off ]"], true)
      end
      tc:GuiUpdate(tc.OptionsName)
      return
   elseif (option == L["threshold"] or option == L["minfree"]) then -- keeping threshold for backwards compatibility
      local numslots = tonumber(arg2)
      if (arg2 == L["off"]) then
         tcDB.useMinFree = false
         chatMsg(L["TC: Trashcan will delete items immediately."], true)
      elseif (numslots and numslots > 0) then
         tcDB.useMinFree = true
         tcDB.MinFree = numslots
         chatMsg(L["TC: Trashcan will try to leave "] .. arg2 .. L[" slots free in you bags."], true)
      else
         chatMsg(L["TC: Valid options are /tc minfree [ <numslots> | off ]"], true)
      end
      tc:GuiUpdate(tc.OptionsName)
      return
   elseif (option == L["remove"]) then
      if (not arg2 or arg2 == "") then
         chatMsg(L["TC: Valid options are /tc remove [ <itemname> | ALL ]"], true)
      elseif (arg2 == L["ALL"]) then
         tcDB.TrashList = {}
         tc:GuiMakeTrashlist()
         chatMsg(L["TC: Your Trashcan list is now empty."], true)
      else
         tcDB.TrashList[arg2] = nil
         tc:GuiMakeTrashlist()
         chatMsg(L["TC: "] .. arg2 .. L[" removed from Trashcan list!"], true)
      end
      tc:GuiUpdate(tc.TrashlistOptionsName)
      return
   elseif (option == L["destroy"]) then
      if (not arg2 or (arg2 ~= "on" and arg2 ~= "off")) then
         chatMsg(L["TC: Valid options are /tc destroy [ on | off ]"], true)
      elseif (arg2=="off") then
         tc.Enable(false, true)
         return
      else
         tc.Enable(true, true)
		end
--    elseif (option == L["off"]) then
-- 		Trashcan:Enable(false, true)
-- 		return
-- 	elseif (option == L["on"]) then
-- 		Trashcan:Enable(true, true)
--  --  The above has been changed to /tc destroy on|off to avoid confusion when selling is still enabled
   elseif (option == L["on"] or option == L["off"]) then
      chatMsg("TC: /tc on | off is no longer a valid option.  Use /tc destroy on | off.", true)
	elseif (option == L["sell"]) then
      if (not arg2 or (arg2 ~= "on" and arg2 ~= "off")) then
         chatMsg(L["TC: Valid options are /tc sell [ on | off ]"], true)
		else
		   tc.Merchant:SellEnable(arg2=="on")
		   tc:GuiUpdate(tc.OptionsName)
		end
		return
   elseif (option == L["test"]) then
		listonly = true
   elseif (option == L["purge"]) then
      local useMinFree = tcDB.useMinFree
      tcDB.useMinFree = false
      destroyJunk()
      tcDB.useMinFree = useMinFree
      return
   else
   	chatMsg(L["TC: Trashcan did not understand that option. Type '/tc help' for valid options."], true)
   	return
	end
   destroyJunk(listonly)
end

function tc:AddCursorToDeleteList()
   if (CursorHasItem()) then
      local itemType, itemId = GetCursorInfo()
      if (itemType and itemType == "item" and itemId) then
         DeleteCursorItem()
         tc:AddToDeleteList("item:" .. itemId)
      end
   end
end

function tc:AddToDeleteList(itemstring)
   if itemstring then
      local _, _, itemId = string.find(itemstring, "item:(%-?%d+)")
      if itemId then
         local name, link, quality, _, _, _, _, _, _, _, value = GetItemInfo(itemId)
         if (name and type(name) == "string") then
            if (quality > 0 or (tcDB.useMaxVal and value > tcDB.MaxVal)) then
               tcDB.TrashList[name] = itemId
               -- If the gui options list has been created, then we need to re-create it.
               tc:GuiMakeTrashlist()
               chatMsg(L["TC: "] .. link .. L[" added to your Trashcan list!"])
               tc:GuiUpdate(tc.TrashlistOptionsName)
               destroyJunk()
               return true
            else
               chatMsg(L["TC: This item would be deleted anyway."])
            end
         end
      end
   end
end

function tc:GSCString(cpval)
   -- Used by options, needs to return a simple string, so don't put it in
   -- one line, please:  parseCopper() returns three other values!!
   local gscString = parseCopper(cpval)
   return gscString
end
